<?php

declare(strict_types=1);

use App\Domain\Product\Models\Product;
use App\Models\User;
use Illuminate\Support\Str;

uses(Illuminate\Foundation\Testing\RefreshDatabase::class);

describe('GraphQL Product API', function () {
    it('returns unauthorized without authentication', function () {
        $response = $this->postJson('/graphql', [
            'query' => '{ product(id: "test-id") { id name } }',
        ]);

        $response->assertOk();
        expect($response->json())->toHaveKey('errors');
    });

    it('queries product by id with authentication', function () {
        $user = User::factory()->create();
        $product = Product::create([
            'id'          => Str::uuid()->toString(),
            'name'        => 'Premium Savings Account',
            'description' => 'High-yield savings product',
            'category'    => 'savings',
            'type'        => 'deposit',
            'status'      => 'active',
        ]);

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/graphql', [
                'query' => '
                    query ($id: ID!) {
                        product(id: $id) {
                            id
                            name
                            description
                            category
                            type
                            status
                        }
                    }
                ',
                'variables' => ['id' => $product->id],
            ]);

        $response->assertOk();
        $data = $response->json('data.product');
        expect($data['name'])->toBe('Premium Savings Account');
        expect($data['category'])->toBe('savings');
        expect($data['status'])->toBe('active');
    });

    it('paginates products', function () {
        $user = User::factory()->create();
        for ($i = 0; $i < 3; $i++) {
            Product::create([
                'id'       => Str::uuid()->toString(),
                'name'     => "Product {$i}",
                'category' => 'lending',
                'type'     => 'loan',
                'status'   => 'active',
            ]);
        }

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/graphql', [
                'query' => '
                    {
                        products(first: 10, page: 1) {
                            data {
                                id
                                name
                                category
                                status
                            }
                            paginatorInfo {
                                total
                            }
                        }
                    }
                ',
            ]);

        $response->assertOk();
        $data = $response->json('data.products');
        expect($data['data'])->toBeArray();
        expect($data['paginatorInfo']['total'])->toBeGreaterThanOrEqual(3);
    });

    it('creates a product via mutation', function () {
        $user = User::factory()->create();

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/graphql', [
                'query' => '
                    mutation ($input: CreateProductInput!) {
                        createProduct(input: $input) {
                            id
                            name
                            category
                            type
                            status
                        }
                    }
                ',
                'variables' => [
                    'input' => [
                        'name'        => 'Business Checking',
                        'description' => 'Business checking account product',
                        'category'    => 'checking',
                        'type'        => 'deposit',
                    ],
                ],
            ]);

        $response->assertOk();
        $json = $response->json();
        expect($json)->not->toHaveKey('errors');
        $data = $response->json('data.createProduct');
        expect($data['name'])->toBe('Business Checking');
        expect($data['category'])->toBe('checking');
    });
});
