<?php

declare(strict_types=1);

use App\Domain\Payment\Models\PaymentTransaction;
use App\Models\User;
use Illuminate\Support\Str;

uses(Illuminate\Foundation\Testing\RefreshDatabase::class);

describe('GraphQL Payment API', function () {
    it('returns unauthorized without authentication', function () {
        $response = $this->postJson('/graphql', [
            'query' => '{ payment(id: 1) { id status } }',
        ]);

        $response->assertOk();
        expect($response->json())->toHaveKey('errors');
    });

    it('queries payment by id with authentication', function () {
        $user = User::factory()->create();
        $payment = PaymentTransaction::create([
            'aggregate_uuid' => Str::uuid()->toString(),
            'account_uuid'   => Str::uuid()->toString(),
            'type'           => 'deposit',
            'status'         => 'pending',
            'amount'         => 10000,
            'currency'       => 'USD',
            'reference'      => 'REF-001',
            'initiated_at'   => now(),
        ]);

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/graphql', [
                'query' => '
                    query ($id: ID!) {
                        payment(id: $id) {
                            id
                            status
                            amount
                            currency
                        }
                    }
                ',
                'variables' => ['id' => $payment->id],
            ]);

        $response->assertOk();
        $data = $response->json('data.payment');
        expect($data['status'])->toBe('pending');
        expect($data['currency'])->toBe('USD');
    });

    it('paginates payments', function () {
        $user = User::factory()->create();
        for ($i = 0; $i < 3; $i++) {
            PaymentTransaction::create([
                'aggregate_uuid' => Str::uuid()->toString(),
                'account_uuid'   => Str::uuid()->toString(),
                'type'           => 'deposit',
                'status'         => 'pending',
                'amount'         => 5000 + $i,
                'currency'       => 'USD',
            ]);
        }

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/graphql', [
                'query' => '
                    {
                        payments(first: 10, page: 1) {
                            data {
                                id
                                status
                                amount
                            }
                            paginatorInfo {
                                total
                            }
                        }
                    }
                ',
            ]);

        $response->assertOk();
        $data = $response->json('data.payments');
        expect($data['data'])->toBeArray();
        expect($data['paginatorInfo']['total'])->toBeGreaterThanOrEqual(3);
    });

    it('initiates a payment via mutation', function () {
        $user = User::factory()->create();
        $accountUuid = Str::uuid()->toString();

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/graphql', [
                'query' => '
                    mutation ($input: InitiatePaymentInput!) {
                        initiatePayment(input: $input) {
                            id
                            status
                            amount
                            currency
                            type
                        }
                    }
                ',
                'variables' => [
                    'input' => [
                        'account_uuid' => $accountUuid,
                        'amount'       => 25000.0,
                        'currency'     => 'EUR',
                        'type'         => 'withdrawal',
                    ],
                ],
            ]);

        $response->assertOk();
        $json = $response->json();
        expect($json)->not->toHaveKey('errors');
        $data = $response->json('data.initiatePayment');
        expect($data['status'])->toBe('pending');
        expect($data['currency'])->toBe('EUR');
    });
});
