<?php

namespace Tests\Feature\Http\Controllers\Api\V2;

use App\Domain\FinancialInstitution\Models\FinancialInstitutionApplication;
use App\Models\User;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Laravel\Sanctum\Sanctum;
use PHPUnit\Framework\Attributes\Test;
use Tests\ControllerTestCase;

class FinancialInstitutionControllerTest extends ControllerTestCase
{
    protected User $user;

    protected function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();
        Storage::fake('private');
    }

    #[Test]
    public function test_get_application_form_returns_structure(): void
    {
        $response = $this->getJson('/api/v2/financial-institutions/application-form');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'data' => [
                    'institution_types',
                    'required_fields' => [
                        'institution_details',
                        'contact_information',
                        'address_information',
                        'business_information',
                        'technical_requirements',
                        'compliance_information',
                    ],
                    'document_requirements',
                ],
            ])
            ->assertJsonPath('data.institution_types.bank', 'Commercial Bank')
            ->assertJsonPath('data.institution_types.credit_union', 'Credit Union');
    }

    #[Test]
    public function test_submit_application_successfully(): void
    {
        Sanctum::actingAs($this->user);

        $applicationData = $this->getValidApplicationData();

        $response = $this->postJson('/api/v2/financial-institutions/apply', $applicationData);

        $response->assertStatus(201)
            ->assertJsonStructure([
                'data' => [
                    'application_id',
                    'application_number',
                    'status',
                    'required_documents',
                    'message',
                ],
            ])
            ->assertJsonPath('data.message', 'Application submitted successfully');

        $this->assertDatabaseHas('financial_institution_applications', [
            'institution_name' => $applicationData['institution_name'],
            'status'           => 'pending',
        ]);
    }

    #[Test]
    public function test_submit_application_validates_required_fields(): void
    {
        Sanctum::actingAs($this->user);

        $response = $this->postJson('/api/v2/financial-institutions/apply', []);

        $response->assertStatus(422)
            ->assertJsonValidationErrors([
                'institution_name',
                'legal_name',
                'registration_number',
            ]);
    }

    #[Test]
    public function test_get_application_status(): void
    {
        Sanctum::actingAs($this->user);

        $application = FinancialInstitutionApplication::factory()->create([
            'status' => 'pending',
        ]);

        $response = $this->getJson("/api/v2/financial-institutions/application/{$application->application_number}/status");

        $response->assertStatus(200)
            ->assertJsonStructure([
                'data' => [
                    'application_number',
                    'institution_name',
                    'status',
                    'review_stage',
                    'risk_rating',
                    'submitted_at',
                    'documents',
                    'is_editable',
                ],
            ])
            ->assertJson([
                'data' => [
                    'application_number' => $application->application_number,
                    'status'             => 'pending',
                ],
            ]);
    }

    #[Test]
    public function test_get_application_status_prevents_unauthorized_access(): void
    {
        // Since the endpoint is public and doesn't check ownership, we'll just verify it returns data
        $application = FinancialInstitutionApplication::factory()->create();

        $response = $this->getJson("/api/v2/financial-institutions/application/{$application->application_number}/status");

        $response->assertStatus(200);
    }

    #[Test]
    public function test_upload_document_successfully(): void
    {
        Sanctum::actingAs($this->user);

        $application = FinancialInstitutionApplication::factory()->create([
            'status' => 'pending',
        ]);

        $file = UploadedFile::fake()->create('license.pdf', 1000);

        $response = $this->postJson("/api/v2/financial-institutions/application/{$application->application_number}/documents", [
            'document_type' => 'regulatory_license',
            'document'      => $file,
            'description'   => 'Banking license from regulatory authority',
        ]);

        $response->assertStatus(200)
            ->assertJsonStructure([
                'data' => [
                    'document_type',
                    'uploaded',
                    'filename',
                    'size',
                    'message',
                ],
            ])
            ->assertJson([
                'data' => [
                    'document_type' => 'regulatory_license',
                    'uploaded'      => true,
                ],
            ]);
    }

    #[Test]
    public function test_upload_document_validates_file_type(): void
    {
        Sanctum::actingAs($this->user);

        $application = FinancialInstitutionApplication::factory()->create([
            'status' => 'pending',
        ]);

        $file = UploadedFile::fake()->create('malicious.exe', 1000);

        $response = $this->postJson("/api/v2/financial-institutions/application/{$application->application_number}/documents", [
            'document_type' => 'regulatory_license',
            'document'      => $file,
        ]);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['document']);
    }

    private function getValidApplicationData(): array
    {
        return [
            // Institution Details
            'institution_name'          => 'Test Bank Ltd',
            'legal_name'                => 'Test Bank Limited',
            'registration_number'       => '12345678',
            'tax_id'                    => 'TB123456',
            'country'                   => 'GB',
            'institution_type'          => 'bank',
            'assets_under_management'   => 1000000000,
            'years_in_operation'        => 10,
            'primary_regulator'         => 'FCA',
            'regulatory_license_number' => 'FCA123456',

            // Contact Information
            'contact_name'       => 'John Doe',
            'contact_email'      => 'john@testbank.com',
            'contact_phone'      => '+441234567890',
            'contact_position'   => 'Chief Compliance Officer',
            'contact_department' => 'Compliance',

            // Address Information
            'headquarters_address'     => '123 Bank Street',
            'headquarters_city'        => 'London',
            'headquarters_state'       => null,
            'headquarters_postal_code' => 'EC1A 1AA',
            'headquarters_country'     => 'GB',

            // Business Information
            'business_description'          => 'Test Bank Limited is a commercial bank providing retail and corporate banking services with over 10 years of experience in the financial sector.',
            'target_markets'                => ['GB', 'EU'],
            'product_offerings'             => ['Deposits', 'Lending', 'Payments'],
            'expected_monthly_transactions' => 10000,
            'expected_monthly_volume'       => 100000000,
            'required_currencies'           => ['EUR', 'USD', 'GBP'],

            // Technical Requirements
            'integration_requirements' => ['API', 'Webhooks', 'Reporting'],
            'requires_api_access'      => true,
            'requires_webhooks'        => true,
            'requires_reporting'       => true,
            'security_certifications'  => ['ISO27001', 'SOC2'],

            // Compliance Information
            'has_aml_program'            => true,
            'has_kyc_procedures'         => true,
            'has_data_protection_policy' => true,
            'is_pci_compliant'           => true,
            'is_gdpr_compliant'          => true,
            'compliance_certifications'  => ['FCA', 'PRA'],
        ];
    }
}
