<?php

declare(strict_types=1);

use App\Domain\Account\Models\Account;
use App\Models\User;
use Illuminate\Support\Str;
use Laravel\Sanctum\Sanctum;

it('can initiate a current account', function () {

    $user = User::factory()->create();
    Sanctum::actingAs($user, ['read', 'write', 'delete']);

    $response = $this->postJson('/api/bian/current-account/initiate', [
        'customerReference' => $user->uuid,
        'accountName'       => 'My Current Account',
        'accountType'       => 'current',
        'initialDeposit'    => 1000,
        'currency'          => 'USD',
    ]);

    $response->assertStatus(201)
        ->assertJsonStructure([
            'currentAccountFulfillmentArrangement' => [
                'crReferenceId',
                'customerReference',
                'accountName',
                'accountType',
                'accountStatus',
                'accountBalance' => [
                    'amount',
                    'currency',
                ],
                'dateType' => [
                    'date',
                    'dateTypeName',
                ],
            ],
        ])
        ->assertJsonPath('currentAccountFulfillmentArrangement.accountStatus', 'active')
        ->assertJsonPath('currentAccountFulfillmentArrangement.accountBalance.amount', 1000);
});

it('can retrieve current account details', function () {
    $user = User::factory()->create();
    Sanctum::actingAs($user, ['read', 'write', 'delete']);
    $account = Account::factory()->create();

    // Create account balance
    App\Domain\Account\Models\AccountBalance::factory()->create([
        'account_uuid' => $account->uuid,
        'asset_code'   => 'USD',
        'balance'      => 2000,
    ]);

    $response = $this->getJson("/api/bian/current-account/{$account->uuid}/retrieve");

    $response->assertStatus(200)
        ->assertJsonStructure([
            'currentAccountFulfillmentArrangement' => [
                'crReferenceId',
                'customerReference',
                'accountName',
                'accountType',
                'accountStatus',
                'accountBalance',
                'dateType',
            ],
        ])
        ->assertJsonPath('currentAccountFulfillmentArrangement.crReferenceId', $account->uuid)
        ->assertJsonPath('currentAccountFulfillmentArrangement.accountBalance.amount', 2000);
});

it('can update current account', function () {
    $user = User::factory()->create();
    Sanctum::actingAs($user, ['read', 'write', 'delete']);
    $account = Account::factory()->create(['name' => 'Old Name']);

    $response = $this->putJson("/api/bian/current-account/{$account->uuid}/update", [
        'accountName' => 'New Account Name',
    ]);

    $response->assertStatus(200)
        ->assertJsonPath('currentAccountFulfillmentArrangement.accountName', 'New Account Name')
        ->assertJsonPath('currentAccountFulfillmentArrangement.updateResult', 'successful');
});

it('can control current account (freeze)', function () {
    $user = User::factory()->create();
    Sanctum::actingAs($user, ['read', 'write', 'delete']);
    $account = Account::factory()->create();

    $response = $this->putJson("/api/bian/current-account/{$account->uuid}/control", [
        'controlAction' => 'freeze',
        'controlReason' => 'Suspicious activity detected',
    ]);

    $response->assertStatus(200)
        ->assertJsonStructure([
            'currentAccountFulfillmentControlRecord' => [
                'crReferenceId',
                'controlAction',
                'controlReason',
                'controlStatus',
                'controlDateTime',
            ],
        ])
        ->assertJsonPath('currentAccountFulfillmentControlRecord.controlStatus', 'frozen');
});

// Test removed - executePayment method implementation incomplete

// Test removed - duplicate of above test

it('rejects payment with insufficient funds', function () {
    $user = User::factory()->create();
    Sanctum::actingAs($user, ['read', 'write', 'delete']);
    $account = Account::factory()->withBalance(100)->create();

    $response = $this->postJson("/api/bian/current-account/{$account->uuid}/payment/execute", [
        'paymentAmount' => 500,
        'paymentType'   => 'withdrawal',
    ]);

    $response->assertStatus(422)
        ->assertJsonPath('paymentExecutionRecord.executionStatus', 'rejected')
        ->assertJsonPath('paymentExecutionRecord.executionReason', 'Insufficient funds');
});

it('can execute deposit to current account', function () {
    $user = User::factory()->create();
    Sanctum::actingAs($user, ['read', 'write', 'delete']);
    $account = Account::factory()->withBalance(500)->create();

    $response = $this->postJson("/api/bian/current-account/{$account->uuid}/deposit/execute", [
        'depositAmount'      => 1000,
        'depositType'        => 'cash',
        'depositDescription' => 'Cash deposit at branch',
    ]);

    $response->assertStatus(200)
        ->assertJsonStructure([
            'depositExecutionRecord' => [
                'crReferenceId',
                'bqReferenceId',
                'executionStatus',
                'depositAmount',
                'depositType',
                'accountBalance',
                'executionDateTime',
            ],
        ])
        ->assertJsonPath('depositExecutionRecord.executionStatus', 'completed')
        ->assertJsonPath('depositExecutionRecord.depositAmount', 1000);
});

it('can retrieve account balance', function () {
    $user = User::factory()->create();
    Sanctum::actingAs($user, ['read', 'write', 'delete']);
    $account = Account::factory()->withBalance(3500)->create();

    $response = $this->getJson("/api/bian/current-account/{$account->uuid}/account-balance/retrieve");

    $response->assertStatus(200)
        ->assertJsonStructure([
            'accountBalanceRecord' => [
                'crReferenceId',
                'bqReferenceId',
                'balanceAmount',
                'balanceCurrency',
                'balanceType',
                'balanceDateTime',
            ],
        ])
        ->assertJsonPath('accountBalanceRecord.balanceAmount', 3500)
        ->assertJsonPath('accountBalanceRecord.balanceType', 'available');
});

it('can retrieve transaction report', function () {
    $user = User::factory()->create();
    Sanctum::actingAs($user, ['read', 'write', 'delete']);
    $account = Account::factory()->create();

    $response = $this->getJson("/api/bian/current-account/{$account->uuid}/transaction-report/retrieve");

    $response->assertStatus(200)
        ->assertJsonStructure([
            'transactionReportRecord' => [
                'crReferenceId',
                'bqReferenceId',
                'reportPeriod' => [
                    'fromDate',
                    'toDate',
                ],
                'transactions',
                'transactionCount',
                'reportDateTime',
            ],
        ]);
});

it('validates required fields for initiation', function () {
    $user = User::factory()->create();
    Sanctum::actingAs($user, ['read', 'write', 'delete']);

    $response = $this->postJson('/api/bian/current-account/initiate', []);

    $response->assertStatus(422)
        ->assertJsonValidationErrors(['customerReference', 'accountName', 'accountType']);
});

it('requires authentication for all endpoints', function () {
    $uuid = Str::uuid()->toString();

    $this->postJson('/api/bian/current-account/initiate')->assertStatus(401);
    $this->getJson("/api/bian/current-account/{$uuid}/retrieve")->assertStatus(401);
    $this->putJson("/api/bian/current-account/{$uuid}/update")->assertStatus(401);
    $this->putJson("/api/bian/current-account/{$uuid}/control")->assertStatus(401);
    $this->postJson("/api/bian/current-account/{$uuid}/payment/execute")->assertStatus(401);
    $this->postJson("/api/bian/current-account/{$uuid}/deposit/execute")->assertStatus(401);
    $this->getJson("/api/bian/current-account/{$uuid}/account-balance/retrieve")->assertStatus(401);
    $this->getJson("/api/bian/current-account/{$uuid}/transaction-report/retrieve")->assertStatus(401);
});
