<?php

use App\Console\Commands\CacheWarmup;
use App\Domain\Account\Models\Account;
use App\Domain\Account\Services\Cache\CacheManager;
use Illuminate\Foundation\Testing\RefreshDatabase;

uses(RefreshDatabase::class);

it('can warm up cache for all accounts', function () {
    // Create test accounts (let's see how many the command actually processes)
    $accounts = Account::factory()->count(3)->create(['frozen' => false]);
    Account::factory()->create(['frozen' => true]); // Frozen account should be skipped

    // Mock the CacheManager to expect calls with any string - let's be more flexible
    $cacheManager = Mockery::mock(CacheManager::class);
    $cacheManager->shouldReceive('warmUp')
        ->withArgs(function ($uuid) {
            return is_string($uuid) || (is_object($uuid) && method_exists($uuid, '__toString'));
        })
        ->atLeast()->times(1) // Allow any number of calls since chunking affects this
        ->andReturn(true);

    $this->app->instance(CacheManager::class, $cacheManager);

    $this->artisan('cache:warmup')
        ->expectsOutput('Warming up cache for all accounts...')
        ->expectsOutput('Cache warmup completed!')
        ->assertExitCode(0);
});

it('can warm up cache for specific accounts', function () {
    $account1 = Account::factory()->create();
    $account2 = Account::factory()->create();

    // Mock the CacheManager to accept string UUIDs
    $cacheManager = Mockery::mock(CacheManager::class);
    $cacheManager->shouldReceive('warmUp')
        ->with((string) $account1->uuid)
        ->once()
        ->andReturn(true);
    $cacheManager->shouldReceive('warmUp')
        ->with((string) $account2->uuid)
        ->once()
        ->andReturn(true);

    $this->app->instance(CacheManager::class, $cacheManager);

    $this->artisan('cache:warmup', [
        '--account' => [(string) $account1->uuid, (string) $account2->uuid],
    ])
        ->expectsOutput("Warming up cache for account: {$account1->uuid}")
        ->expectsOutput("Warming up cache for account: {$account2->uuid}")
        ->expectsOutput('Cache warmup completed!')
        ->assertExitCode(0);
});

it('has correct signature and description', function () {
    $command = app(CacheWarmup::class);

    expect($command->getName())->toBe('cache:warmup');
    expect($command->getDescription())->toBe('Warm up Redis cache for accounts');
});
