<?php

declare(strict_types=1);

namespace App\Domain\Security\Checks;

use App\Domain\Security\Contracts\SecurityCheckInterface;
use App\Domain\Security\ValueObjects\SecurityCheckResult;
use Symfony\Component\Process\Process;
use Throwable;

/**
 * A06: Vulnerable and Outdated Components.
 *
 * Runs `composer audit --format=json` to check for known vulnerabilities.
 */
class DependencyVulnerabilityCheck implements SecurityCheckInterface
{
    public function getName(): string
    {
        return 'dependency_vulnerability';
    }

    public function getCategory(): string
    {
        return 'A06: Vulnerable and Outdated Components';
    }

    public function run(): SecurityCheckResult
    {
        $findings = [];
        $recommendations = [];

        try {
            $process = new Process(['composer', 'audit', '--format=json', '--no-interaction']);
            $process->setTimeout(60);
            $process->setWorkingDirectory(base_path());
            $process->run();

            $output = $process->getOutput();
            $data = json_decode($output, true);

            if (! empty($data['advisories'])) {
                foreach ($data['advisories'] as $package => $advisories) {
                    foreach ($advisories as $advisory) {
                        $findings[] = sprintf(
                            '%s: %s (CVE: %s)',
                            $package,
                            $advisory['title'] ?? 'Unknown vulnerability',
                            $advisory['cve'] ?? 'N/A',
                        );
                    }
                }
            }

            if (! $process->isSuccessful() && empty($findings)) {
                $findings[] = 'composer audit returned non-zero exit code';
            }
        } catch (Throwable $e) {
            $findings[] = 'Could not run composer audit: ' . $e->getMessage();
            $recommendations[] = 'Ensure composer is installed and accessible';
        }

        if (! empty($findings)) {
            $recommendations[] = 'Run `composer audit` and update affected packages';
            $recommendations[] = 'Review security advisories at https://packagist.org/advisories';
        }

        $score = empty($findings) ? 100 : max(0, 100 - (count($findings) * 20));

        return new SecurityCheckResult(
            name: $this->getName(),
            category: $this->getCategory(),
            passed: empty($findings),
            score: $score,
            findings: $findings,
            recommendations: $recommendations,
            severity: empty($findings) ? 'info' : 'high',
        );
    }
}
