<?php

declare(strict_types=1);

namespace App\Domain\AgentProtocol\Services;

use App\Domain\AgentProtocol\Contracts\TransactionVerifierInterface;
use App\Domain\AgentProtocol\Models\Agent;
use App\Domain\AgentProtocol\Models\AgentTransaction;
use Exception;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

/**
 * Service for comprehensive transaction verification.
 *
 * Configuration is loaded from config/agent_protocol.php:
 * - verification.levels: Verification level definitions and their checks
 * - verification.risk_thresholds: Score thresholds for risk levels
 * - verification.risk_weights: Weights for risk calculation by check type
 * - verification.velocity_limits: Transaction velocity limits by period
 * - verification.default_limits: Default transaction limits
 * - verification.cache_ttl: Cache duration for verification results
 * - verification.multi_factor_threshold: Required MFA factors
 */
class TransactionVerificationService implements TransactionVerifierInterface
{
    public function __construct(
        private readonly DigitalSignatureService $signatureService,
        private readonly EncryptionService $encryptionService,
        private readonly FraudDetectionService $fraudService
    ) {
    }

    /**
     * Verify a transaction and return the verification result.
     *
     * {@inheritDoc}
     */
    public function verify(string $transactionId, array $transactionData): array
    {
        $senderId = $transactionData['sender_id'] ?? '';
        $result = $this->verifyTransaction(
            $transactionId,
            $senderId,
            $transactionData,
            $transactionData['metadata'] ?? [],
            'standard'
        );

        // Map to interface format
        $checksPassed = [];
        $checksFailed = [];

        foreach ($result['checks'] ?? [] as $checkName => $checkResult) {
            if ($checkResult['passed'] ?? false) {
                $checksPassed[] = $checkName;
            } else {
                $checksFailed[] = $checkName;
            }
        }

        return [
            'verified'           => $result['status'] === 'approved',
            'verification_level' => $result['verification_level'] ?? 'standard',
            'risk_score'         => (int) ($result['risk_score'] ?? 0),
            'checks_passed'      => $checksPassed,
            'checks_failed'      => $checksFailed,
            'timestamp'          => $result['timestamp'] ?? now()->toIso8601String(),
        ];
    }

    /**
     * Get the verification level for a given risk score.
     *
     * {@inheritDoc}
     */
    public function getVerificationLevel(int $riskScore): string
    {
        return $this->determineRiskLevel((float) $riskScore);
    }

    /**
     * Calculate the risk score for a transaction.
     *
     * {@inheritDoc}
     */
    public function calculateRiskScore(array $transactionData): int
    {
        $senderId = $transactionData['sender_id'] ?? '';
        $amount = (float) ($transactionData['amount'] ?? 0);

        // Perform fraud analysis
        $fraudAnalysis = $this->fraudService->analyzeTransaction(
            'risk_calc_' . uniqid(),
            $senderId,
            $amount,
            $transactionData
        );

        return (int) round($fraudAnalysis['risk_score'] ?? 0);
    }

    /**
     * Check if a transaction passes velocity limits.
     *
     * {@inheritDoc}
     */
    public function checkVelocityLimits(string $agentId, float $amount): array
    {
        $result = $this->performVelocityChecks($agentId, $amount);
        $velocityLimits = $this->getVelocityLimits();
        $stats = $this->getTransactionStats($agentId, 'daily');

        return [
            'passed'       => $result['passed'],
            'hourly_count' => $this->getTransactionStats($agentId, 'hourly')['count'],
            'daily_count'  => $stats['count'],
            'daily_amount' => (float) $stats['total_amount'],
            'limits'       => [
                'hourly_count'  => $velocityLimits['hourly']['count'],
                'daily_count'   => $velocityLimits['daily']['count'],
                'hourly_amount' => $velocityLimits['hourly']['limit'],
                'daily_amount'  => $velocityLimits['daily']['limit'],
            ],
        ];
    }

    /**
     * Get verification levels from configuration.
     *
     * @return array<string, array<string>>
     */
    private function getVerificationLevels(): array
    {
        return config('agent_protocol.verification.levels', [
            'basic'    => ['signature', 'agent'],
            'standard' => ['signature', 'agent', 'limits', 'velocity'],
            'enhanced' => ['signature', 'agent', 'limits', 'velocity', 'fraud', 'compliance'],
            'maximum'  => ['signature', 'agent', 'limits', 'velocity', 'fraud', 'compliance', 'encryption', 'multi_factor'],
        ]);
    }

    /**
     * Get risk thresholds from configuration.
     *
     * @return array<string, int>
     */
    private function getRiskThresholds(): array
    {
        $thresholds = config('agent_protocol.verification.risk_thresholds', []);

        return [
            'low'      => (int) ($thresholds['low'] ?? 30),
            'medium'   => (int) ($thresholds['medium'] ?? 60),
            'high'     => (int) ($thresholds['high'] ?? 80),
            'critical' => (int) ($thresholds['critical'] ?? 95),
        ];
    }

    /**
     * Get risk weights from configuration.
     *
     * @return array<string, int>
     */
    private function getRiskWeights(): array
    {
        $weights = config('agent_protocol.verification.risk_weights', []);

        return [
            'signature'    => (int) ($weights['signature'] ?? 25),
            'agent'        => (int) ($weights['agent'] ?? 15),
            'limits'       => (int) ($weights['limits'] ?? 10),
            'velocity'     => (int) ($weights['velocity'] ?? 15),
            'fraud'        => (int) ($weights['fraud'] ?? 20),
            'compliance'   => (int) ($weights['compliance'] ?? 10),
            'encryption'   => (int) ($weights['encryption'] ?? 3),
            'multi_factor' => (int) ($weights['multi_factor'] ?? 2),
        ];
    }

    /**
     * Get velocity limits from configuration.
     *
     * @return array<string, array{limit: int, count: int}>
     */
    private function getVelocityLimits(): array
    {
        $limits = config('agent_protocol.verification.velocity_limits', []);

        return [
            'hourly' => [
                'limit' => (int) ($limits['hourly']['amount'] ?? 10000),
                'count' => (int) ($limits['hourly']['count'] ?? 10),
            ],
            'daily' => [
                'limit' => (int) ($limits['daily']['amount'] ?? 50000),
                'count' => (int) ($limits['daily']['count'] ?? 50),
            ],
            'weekly' => [
                'limit' => (int) ($limits['weekly']['amount'] ?? 200000),
                'count' => (int) ($limits['weekly']['count'] ?? 200),
            ],
            'monthly' => [
                'limit' => (int) ($limits['monthly']['amount'] ?? 500000),
                'count' => (int) ($limits['monthly']['count'] ?? 500),
            ],
        ];
    }

    /**
     * Get default transaction limits from configuration.
     *
     * @return array<string, int>
     */
    private function getDefaultLimits(): array
    {
        $limits = config('agent_protocol.verification.default_limits', []);

        return [
            'single_transaction' => (int) ($limits['single_transaction'] ?? 10000),
            'daily'              => (int) ($limits['daily'] ?? 50000),
            'monthly'            => (int) ($limits['monthly'] ?? 200000),
        ];
    }

    /**
     * Perform comprehensive transaction verification.
     *
     * @param string $transactionId Unique transaction identifier
     * @param string $agentId Agent's DID
     * @param array<string, mixed> $transactionData Transaction data to verify
     * @param array<string, mixed> $securityMetadata Security-related metadata
     * @param string $verificationLevel Verification level (basic, standard, enhanced, maximum)
     * @return array<string, mixed> Verification results
     */
    public function verifyTransaction(
        string $transactionId,
        string $agentId,
        array $transactionData,
        array $securityMetadata,
        string $verificationLevel = 'standard'
    ): array {
        try {
            $verificationResults = [
                'transaction_id'     => $transactionId,
                'agent_id'           => $agentId,
                'verification_level' => $verificationLevel,
                'timestamp'          => now()->toIso8601String(),
                'checks'             => [],
            ];

            // Get verification checks for level from config
            $verificationLevels = $this->getVerificationLevels();
            $requiredChecks = $verificationLevels[$verificationLevel] ?? $verificationLevels['standard'];

            // Perform each verification check
            foreach ($requiredChecks as $check) {
                $result = $this->performCheck($check, $transactionId, $agentId, $transactionData, $securityMetadata);
                $verificationResults['checks'][$check] = $result;

                // Stop on critical failures
                if (! $result['passed'] && $result['severity'] === 'critical') {
                    $verificationResults['status'] = 'rejected';
                    $verificationResults['reason'] = "Critical failure in {$check} check";
                    break;
                }
            }

            // Calculate overall verification status
            if (! isset($verificationResults['status'])) {
                $verificationResults = $this->calculateVerificationStatus($verificationResults);
            }

            // Calculate risk score
            $verificationResults['risk_score'] = $this->calculateVerificationRiskScore($verificationResults['checks']);
            $verificationResults['risk_level'] = $this->determineRiskLevel($verificationResults['risk_score']);

            // Store verification result
            $this->storeVerificationResult($transactionId, $verificationResults);

            // Log verification
            Log::info('Transaction verification completed', [
                'transaction_id' => $transactionId,
                'status'         => $verificationResults['status'],
                'risk_level'     => $verificationResults['risk_level'],
            ]);

            return $verificationResults;
        } catch (Exception $e) {
            Log::error('Transaction verification failed', [
                'transaction_id' => $transactionId,
                'agent_id'       => $agentId,
                'error'          => $e->getMessage(),
            ]);

            return [
                'transaction_id' => $transactionId,
                'status'         => 'error',
                'error'          => $e->getMessage(),
                'timestamp'      => now()->toIso8601String(),
            ];
        }
    }

    /**
     * Verify transaction integrity with hash validation.
     */
    public function verifyTransactionIntegrity(
        string $transactionId,
        array $transactionData,
        string $expectedHash
    ): bool {
        try {
            // Calculate transaction hash
            $actualHash = $this->calculateTransactionHash($transactionData);

            // Compare hashes
            $isValid = hash_equals($expectedHash, $actualHash);

            // Check for tampering indicators
            if ($isValid) {
                $isValid = $this->checkForTampering($transactionId, $transactionData);
            }

            Log::info('Transaction integrity check', [
                'transaction_id' => $transactionId,
                'is_valid'       => $isValid,
            ]);

            return $isValid;
        } catch (Exception $e) {
            Log::error('Transaction integrity verification failed', [
                'transaction_id' => $transactionId,
                'error'          => $e->getMessage(),
            ]);

            return false;
        }
    }

    /**
     * Verify transaction compliance with regulatory requirements.
     */
    public function verifyCompliance(
        string $transactionId,
        string $agentId,
        array $transactionData
    ): array {
        try {
            $complianceChecks = [];

            // KYC/KYB verification
            $complianceChecks['kyc'] = $this->verifyKYCStatus($agentId);

            // Transaction limits
            $limitsResult = $this->verifyTransactionLimits(
                $agentId,
                $transactionData['amount'] ?? 0,
                $transactionData['currency'] ?? 'USD'
            );
            $complianceChecks['limits'] = [
                'passed'  => $limitsResult['within_limits'],
                'reason'  => $limitsResult['reason'] ?? null,
                'details' => $limitsResult,
            ];

            // Sanctions screening
            $complianceChecks['sanctions'] = $this->checkSanctionsList($agentId);

            // AML checks
            $complianceChecks['aml'] = $this->performAMLChecks($transactionData);

            // Regulatory reporting requirements
            $complianceChecks['reporting'] = $this->checkReportingRequirements(
                $transactionData['amount'] ?? 0,
                $transactionData['currency'] ?? 'USD'
            );

            $isCompliant = array_reduce($complianceChecks, function ($carry, $check) {
                return $carry && $check['passed'];
            }, true);

            return [
                'is_compliant'   => $isCompliant,
                'checks'         => $complianceChecks,
                'transaction_id' => $transactionId,
                'timestamp'      => now()->toIso8601String(),
            ];
        } catch (Exception $e) {
            Log::error('Compliance verification failed', [
                'transaction_id' => $transactionId,
                'error'          => $e->getMessage(),
            ]);

            return [
                'is_compliant'   => false,
                'error'          => $e->getMessage(),
                'transaction_id' => $transactionId,
            ];
        }
    }

    /**
     * Perform velocity checks for fraud prevention using configured limits.
     *
     * @param string $agentId Agent's DID
     * @param float $amount Transaction amount
     * @return array<string, mixed> Velocity check results
     */
    public function performVelocityChecks(string $agentId, float $amount): array
    {
        try {
            $velocityLimits = $this->getVelocityLimits();
            $violations = [];

            foreach ($velocityLimits as $period => $limits) {
                $stats = $this->getTransactionStats($agentId, $period);

                if ($stats['total_amount'] + $amount > $limits['limit']) {
                    $violations[] = [
                        'type'    => 'amount_limit',
                        'period'  => $period,
                        'current' => $stats['total_amount'],
                        'limit'   => $limits['limit'],
                    ];
                }

                if ($stats['count'] >= $limits['count']) {
                    $violations[] = [
                        'type'    => 'count_limit',
                        'period'  => $period,
                        'current' => $stats['count'],
                        'limit'   => $limits['count'],
                    ];
                }
            }

            return [
                'passed'     => empty($violations),
                'violations' => $violations,
                'agent_id'   => $agentId,
                'timestamp'  => now()->toIso8601String(),
            ];
        } catch (Exception $e) {
            Log::error('Velocity check failed', [
                'agent_id' => $agentId,
                'error'    => $e->getMessage(),
            ]);

            return [
                'passed' => false,
                'error'  => $e->getMessage(),
            ];
        }
    }

    /**
     * Verify multi-factor authentication for high-value transactions.
     */
    public function verifyMultiFactor(
        string $agentId,
        string $transactionId,
        array $factors
    ): array {
        try {
            $verifiedFactors = [];

            // Verify each authentication factor
            foreach ($factors as $factorType => $factorData) {
                $verifiedFactors[$factorType] = $this->verifyAuthFactor(
                    $agentId,
                    $factorType,
                    $factorData
                );
            }

            // Require at least 2 factors for high-value transactions
            $verifiedCount = count(array_filter($verifiedFactors, fn ($v) => $v['verified']));
            $isVerified = $verifiedCount >= 2;

            return [
                'verified'       => $isVerified,
                'factors'        => $verifiedFactors,
                'verified_count' => $verifiedCount,
                'required_count' => 2,
                'transaction_id' => $transactionId,
                'timestamp'      => now()->toIso8601String(),
            ];
        } catch (Exception $e) {
            Log::error('Multi-factor verification failed', [
                'transaction_id' => $transactionId,
                'error'          => $e->getMessage(),
            ]);

            return [
                'verified' => false,
                'error'    => $e->getMessage(),
            ];
        }
    }

    /**
     * Perform a specific verification check.
     */
    private function performCheck(
        string $checkType,
        string $transactionId,
        string $agentId,
        array $transactionData,
        array $securityMetadata
    ): array {
        return match ($checkType) {
            'signature'    => $this->verifySignatureCheck($transactionId, $agentId, $transactionData, $securityMetadata),
            'agent'        => $this->verifyAgentCheck($agentId),
            'limits'       => $this->verifyLimitsCheck($agentId, $transactionData['amount'] ?? 0),
            'velocity'     => $this->verifyVelocityCheck($agentId, $transactionData['amount'] ?? 0),
            'fraud'        => $this->verifyFraudCheck($transactionId, $agentId, $transactionData),
            'compliance'   => $this->verifyComplianceCheck($transactionId, $agentId, $transactionData),
            'encryption'   => $this->verifyEncryptionCheck($securityMetadata),
            'multi_factor' => $this->verifyMultiFactorCheck($agentId, $transactionId, $securityMetadata),
            default        => ['passed' => false, 'reason' => 'Unknown check type'],
        };
    }

    /**
     * Verify digital signature.
     */
    private function verifySignatureCheck(
        string $transactionId,
        string $agentId,
        array $transactionData,
        array $metadata
    ): array {
        if (! isset($metadata['signature'])) {
            return ['passed' => false, 'reason' => 'Missing signature', 'severity' => 'critical'];
        }

        $result = $this->signatureService->verifyAgentSignature(
            $transactionId,
            $agentId,
            $transactionData,
            $metadata['signature'],
            $metadata
        );

        return [
            'passed'   => $result['is_valid'],
            'reason'   => $result['reason'] ?? null,
            'severity' => $result['is_valid'] ? 'info' : 'critical',
        ];
    }

    /**
     * Verify agent status and eligibility.
     */
    private function verifyAgentCheck(string $agentId): array
    {
        $agent = Agent::where('agent_id', $agentId)->first();

        if (! $agent) {
            return ['passed' => false, 'reason' => 'Agent not found', 'severity' => 'critical'];
        }

        if ($agent->status !== 'active') {
            return ['passed' => false, 'reason' => 'Agent not active', 'severity' => 'high'];
        }

        if ($agent->is_suspended) {
            return ['passed' => false, 'reason' => 'Agent suspended', 'severity' => 'critical'];
        }

        return ['passed' => true, 'severity' => 'info'];
    }

    /**
     * Verify transaction limits.
     */
    private function verifyLimitsCheck(string $agentId, float $amount): array
    {
        $result = $this->verifyTransactionLimits($agentId, $amount, 'USD');

        return [
            'passed'   => $result['within_limits'],
            'reason'   => $result['reason'] ?? null,
            'severity' => $result['within_limits'] ? 'info' : 'high',
            'details'  => $result,
        ];
    }

    /**
     * Verify velocity limits.
     */
    private function verifyVelocityCheck(string $agentId, float $amount): array
    {
        $result = $this->performVelocityChecks($agentId, $amount);

        return [
            'passed'     => $result['passed'],
            'reason'     => ! $result['passed'] ? 'Velocity limit exceeded' : null,
            'severity'   => $result['passed'] ? 'info' : 'medium',
            'violations' => $result['violations'] ?? [],
        ];
    }

    /**
     * Verify fraud detection.
     */
    private function verifyFraudCheck(
        string $transactionId,
        string $agentId,
        array $transactionData
    ): array {
        $fraudAnalysis = $this->fraudService->analyzeTransaction(
            $transactionId,
            $agentId,
            $transactionData['amount'] ?? 0,
            $transactionData
        );

        $passed = $fraudAnalysis['decision'] !== 'reject';

        return [
            'passed'       => $passed,
            'reason'       => ! $passed ? 'High fraud risk detected' : null,
            'severity'     => $passed ? 'info' : 'high',
            'risk_score'   => $fraudAnalysis['risk_score'],
            'risk_factors' => $fraudAnalysis['risk_factors'] ?? [],
        ];
    }

    /**
     * Verify compliance requirements.
     */
    private function verifyComplianceCheck(
        string $transactionId,
        string $agentId,
        array $transactionData
    ): array {
        $complianceResult = $this->verifyCompliance($transactionId, $agentId, $transactionData);

        return [
            'passed'   => $complianceResult['is_compliant'],
            'reason'   => ! $complianceResult['is_compliant'] ? 'Compliance check failed' : null,
            'severity' => $complianceResult['is_compliant'] ? 'info' : 'critical',
            'details'  => $complianceResult['checks'] ?? [],
        ];
    }

    /**
     * Verify encryption status.
     */
    private function verifyEncryptionCheck(array $metadata): array
    {
        if (! isset($metadata['encrypted']) || ! $metadata['encrypted']) {
            return [
                'passed'   => false,
                'reason'   => 'Transaction not encrypted',
                'severity' => 'medium',
            ];
        }

        // Verify the encryption algorithm is supported
        $supportedCipher = $this->encryptionService->isCipherSupported($metadata['cipher'] ?? '');

        return [
            'passed'    => true,
            'cipher'    => $metadata['cipher'] ?? 'unknown',
            'supported' => $supportedCipher,
            'severity'  => 'info',
        ];
    }

    /**
     * Verify multi-factor authentication.
     */
    private function verifyMultiFactorCheck(
        string $agentId,
        string $transactionId,
        array $metadata
    ): array {
        if (! isset($metadata['auth_factors'])) {
            return [
                'passed'   => false,
                'reason'   => 'Multi-factor authentication required',
                'severity' => 'high',
            ];
        }

        $mfaResult = $this->verifyMultiFactor($agentId, $transactionId, $metadata['auth_factors']);

        return [
            'passed'           => $mfaResult['verified'],
            'reason'           => ! $mfaResult['verified'] ? 'Multi-factor authentication failed' : null,
            'severity'         => $mfaResult['verified'] ? 'info' : 'high',
            'verified_factors' => $mfaResult['verified_count'],
        ];
    }

    /**
     * Calculate overall verification status.
     */
    private function calculateVerificationStatus(array $results): array
    {
        $failedCritical = false;
        $failedHigh = false;
        $warnings = 0;

        foreach ($results['checks'] as $check) {
            if (! $check['passed']) {
                if ($check['severity'] === 'critical') {
                    $failedCritical = true;
                } elseif ($check['severity'] === 'high') {
                    $failedHigh = true;
                } elseif ($check['severity'] === 'medium') {
                    $warnings++;
                }
            }
        }

        if ($failedCritical) {
            $results['status'] = 'rejected';
            $results['reason'] = 'Critical verification failure';
        } elseif ($failedHigh) {
            $results['status'] = 'review_required';
            $results['reason'] = 'High severity issues detected';
        } elseif ($warnings > 2) {
            $results['status'] = 'review_required';
            $results['reason'] = 'Multiple warnings detected';
        } else {
            $results['status'] = 'approved';
        }

        return $results;
    }

    /**
     * Calculate risk score based on verification checks using configured weights.
     *
     * @param array<string, array<string, mixed>> $checks Verification check results
     * @return float Risk score (0-100)
     */
    private function calculateVerificationRiskScore(array $checks): float
    {
        $baseScore = 0;
        $weights = $this->getRiskWeights();

        foreach ($checks as $checkType => $result) {
            if (! $result['passed']) {
                $weight = $weights[$checkType] ?? 5;
                $severityMultiplier = match ($result['severity'] ?? 'medium') {
                    'critical' => 2.0,
                    'high'     => 1.5,
                    'medium'   => 1.0,
                    'low'      => 0.5,
                    default    => 1.0,
                };
                $baseScore += $weight * $severityMultiplier;
            }
        }

        return min(100, $baseScore);
    }

    /**
     * Determine risk level from score using configured thresholds.
     *
     * @param float $score Risk score (0-100)
     * @return string Risk level: 'low', 'medium', 'high', or 'critical'
     */
    private function determineRiskLevel(float $score): string
    {
        $thresholds = $this->getRiskThresholds();

        if ($score <= $thresholds['low']) {
            return 'low';
        } elseif ($score <= $thresholds['medium']) {
            return 'medium';
        } elseif ($score <= $thresholds['high']) {
            return 'high';
        }

        return 'critical';
    }

    /**
     * Calculate transaction hash for integrity verification.
     */
    private function calculateTransactionHash(array $data): string
    {
        ksort($data);
        $json = json_encode($data);

        return hash('sha256', $json !== false ? $json : '');
    }

    /**
     * Check for transaction tampering.
     */
    private function checkForTampering(string $transactionId, array $data): bool
    {
        // Check if transaction exists in database
        $transaction = AgentTransaction::where('transaction_id', $transactionId)->first();
        if (! $transaction) {
            return false;
        }

        // Compare key fields
        if (isset($data['amount']) && $transaction->amount != $data['amount']) {
            return false;
        }

        if (isset($data['recipient']) && $transaction->to_agent_id != $data['recipient']) {
            return false;
        }

        return true;
    }

    /**
     * Verify KYC status.
     */
    private function verifyKYCStatus(string $agentId): array
    {
        $agent = Agent::where('agent_id', $agentId)->first();

        return [
            'passed'      => $agent && $agent->kyc_verified,
            'status'      => $agent ? $agent->kyc_status : 'unknown',
            'verified_at' => $agent?->kyc_verified_at,
        ];
    }

    /**
     * Verify transaction limits using agent-specific or configured default limits.
     *
     * @param string $agentId Agent's DID
     * @param float $amount Transaction amount
     * @param string $currency Currency code
     * @return array<string, mixed> Limit verification result
     */
    private function verifyTransactionLimits(string $agentId, float $amount, string $currency): array
    {
        $agent = Agent::where('agent_id', $agentId)->first();

        if (! $agent) {
            return ['within_limits' => false, 'reason' => 'Agent not found'];
        }

        $defaultLimits = $this->getDefaultLimits();
        $limits = [
            'single_transaction' => $agent->single_transaction_limit ?? $defaultLimits['single_transaction'],
            'daily'              => $agent->daily_limit ?? $defaultLimits['daily'],
            'monthly'            => $agent->monthly_limit ?? $defaultLimits['monthly'],
        ];

        // Convert amount if needed
        $convertedAmount = $this->convertCurrency($amount, $currency, 'USD');

        if ($convertedAmount > $limits['single_transaction']) {
            return [
                'within_limits' => false,
                'reason'        => 'Exceeds single transaction limit',
                'limit'         => $limits['single_transaction'],
                'amount'        => $convertedAmount,
            ];
        }

        // Check daily limit
        $dailyTotal = $this->getTransactionStats($agentId, 'daily')['total_amount'];
        if ($dailyTotal + $convertedAmount > $limits['daily']) {
            return [
                'within_limits' => false,
                'reason'        => 'Exceeds daily limit',
                'limit'         => $limits['daily'],
                'current'       => $dailyTotal,
            ];
        }

        return ['within_limits' => true];
    }

    /**
     * Check sanctions list.
     */
    private function checkSanctionsList(string $agentId): array
    {
        // This would integrate with sanctions screening APIs
        // For now, return mock result
        return [
            'passed'        => true,
            'checked_lists' => ['OFAC', 'EU', 'UN'],
            'matches'       => [],
        ];
    }

    /**
     * Perform AML checks.
     */
    private function performAMLChecks(array $transactionData): array
    {
        $suspicious = false;
        $indicators = [];

        // Check for AML red flags
        if (($transactionData['amount'] ?? 0) > 9999) {
            $indicators[] = 'Large transaction amount';
        }

        if (isset($transactionData['is_cash']) && $transactionData['is_cash']) {
            $indicators[] = 'Cash transaction';
        }

        if (isset($transactionData['high_risk_country']) && $transactionData['high_risk_country']) {
            $indicators[] = 'High-risk jurisdiction';
            $suspicious = true;
        }

        return [
            'passed'       => ! $suspicious,
            'indicators'   => $indicators,
            'requires_sar' => count($indicators) > 2,
        ];
    }

    /**
     * Check reporting requirements.
     */
    private function checkReportingRequirements(float $amount, string $currency): array
    {
        $usdAmount = $this->convertCurrency($amount, $currency, 'USD');

        return [
            'passed'       => true,
            'ctr_required' => $usdAmount >= 10000,
            'sar_required' => false,
            'amount_usd'   => $usdAmount,
        ];
    }

    /**
     * Get transaction statistics for a period.
     */
    private function getTransactionStats(string $agentId, string $period): array
    {
        $startDate = match ($period) {
            'hourly'  => now()->subHour(),
            'daily'   => now()->subDay(),
            'weekly'  => now()->subWeek(),
            'monthly' => now()->subMonth(),
            default   => now()->subDay(),
        };

        $stats = DB::table('agent_transactions')
            ->where('from_agent_id', $agentId)
            ->where('created_at', '>=', $startDate)
            ->where('status', 'completed')
            ->selectRaw('COUNT(*) as count, SUM(amount) as total_amount')
            ->first();

        return [
            'count'        => $stats->count ?? 0,
            'total_amount' => $stats->total_amount ?? 0,
        ];
    }

    /**
     * Verify authentication factor.
     */
    private function verifyAuthFactor(string $agentId, string $factorType, array $factorData): array
    {
        // This would integrate with actual authentication systems
        // For now, return mock verification
        return match ($factorType) {
            'password'     => ['verified' => true, 'type' => 'password'],
            'totp'         => ['verified' => true, 'type' => 'totp'],
            'biometric'    => ['verified' => true, 'type' => 'biometric'],
            'hardware_key' => ['verified' => true, 'type' => 'hardware_key'],
            default        => ['verified' => false, 'type' => $factorType],
        };
    }

    /**
     * Convert currency (simplified).
     */
    private function convertCurrency(float $amount, string $from, string $to): float
    {
        if ($from === $to) {
            return $amount;
        }

        // This would use real exchange rates
        // For now, use simplified conversion
        $rates = [
            'EUR' => 1.1,
            'GBP' => 1.3,
            'JPY' => 0.009,
        ];

        if ($from === 'USD') {
            return $amount / ($rates[$to] ?? 1);
        } elseif ($to === 'USD') {
            return $amount * ($rates[$from] ?? 1);
        }

        return $amount;
    }

    /**
     * Store verification result.
     */
    private function storeVerificationResult(string $transactionId, array $result): void
    {
        Cache::put(
            "verification:{$transactionId}",
            $result,
            now()->addDays(30)
        );
    }
}
