# v2.9.0 Phase 1: ML Anomaly Detection Implementation

> **Released**: February 9, 2026 | **PRs**: #415-#428 | **Tests**: 115 fraud domain tests

## Architecture

The anomaly detection system extends the existing Fraud domain with 4 detection categories coordinated by an orchestrator.

### Flow
```
FraudDetectionService::analyzeTransaction()
  -> ruleEngine->evaluate()
  -> behavioralAnalysis->analyze()
  -> deviceService->analyzeDevice()
  -> anomalyOrchestrator->detectAnomalies()     # NEW
      -> runStatisticalDetection()
      -> runBehavioralDetection()
      -> runVelocityDetection()
      -> runGeolocationDetection()
      -> Persist to anomaly_detections (if score >= 25)
      -> Fire AnomalyDetected event
  -> mlService->predict()
  -> calculateTotalScore()
```

### Key Files

| File | Purpose |
|------|---------|
| `app/Domain/Fraud/Services/AnomalyDetectionOrchestrator.php` | Coordinates all 4 detection types, sanitizes input, persists results |
| `app/Domain/Fraud/Services/StatisticalAnalysisService.php` | Z-score, IQR, Isolation Forest, LOF, Seasonal decomposition |
| `app/Domain/Fraud/Services/GeoMathService.php` | Haversine distance, DBSCAN clustering, impossible travel |
| `app/Domain/Fraud/Models/AnomalyDetection.php` | Polymorphic model with UUID, severity calculation, feedback recording |
| `app/Domain/Fraud/Events/AnomalyDetected.php` | Event fired when anomaly persisted |
| `app/Domain/Fraud/Jobs/ProcessAnomalyBatchJob.php` | Batch scanning of transactions |
| `config/fraud.php` | All fraud ML settings (thresholds, windows, limits) |

### Enums
- `AnomalyType`: statistical, behavioral, velocity, geolocation
- `AnomalyStatus`: detected, investigating, confirmed, false_positive, resolved (with isTerminal())
- `DetectionMethod`: z_score, iqr, isolation_forest, lof, adaptive_threshold, drift_detection, sliding_window, burst_detection, impossible_travel, ip_reputation, geo_clustering

### Database Tables

**`anomaly_detections`** (UUID PK, soft deletes):
- Polymorphic: entity_id/entity_type
- Classification: anomaly_type, detection_method, status, severity
- Scores: anomaly_score (decimal 5,2), confidence (decimal 5,4)
- JSON payloads: features, thresholds, explanation, raw_scores, context_snapshot, baseline_snapshot
- Links: fraud_score_id, fraud_case_id (indexed)
- Audit: reviewed_by (FK), reviewed_at, previous_status
- Indexes: (entity_id, entity_type), (anomaly_type, created_at), (user_id, anomaly_type, created_at), (severity, status), pipeline_run_id

**`behavioral_profiles`** additions:
- adaptive_thresholds, segment_tags, drift_metrics, seasonal_patterns, sliding_window_stats (JSON)
- user_segment (nullable string), drift_score (nullable decimal), last_drift_check_at (nullable datetime)

### Security Hardening (PR #425)
- PII protection: IP addresses stored as SHA-256 hashes in context_snapshot, lat/lon excluded
- Input sanitization: `sanitizeContext()` clamps lat/lon to [-90,90]/[-180,180], rejects negative amounts
- DoS prevention: DBSCAN max_points=1000, transaction_history max_history_size=1000
- Config defaults: enabled=false, demo_mode=false
- High-risk countries moved from hardcoded to config

### Config Reference (`config/fraud.php`)
```php
'anomaly_detection' => ['enabled' => false, 'demo_mode' => false]
'statistical' => [
    'z_score_threshold' => 3.0,
    'iqr_multiplier' => 1.5,
    'isolation_forest_contamination' => 0.1,
    'lof_neighbors' => 20,
    'history_days' => 90,
    'min_samples' => 10,
    'max_history_size' => 1000,
]
'behavioral' => [
    'adaptive_sensitivity' => 1.5,
    'drift_window_days' => 7,
    'drift_baseline_days' => 90,
    'drift_threshold' => 0.3,
    'segments' => [high_value_trader, retail_consumer, occasional_user, new_account, dormant_reactivated],
]
'velocity' => [
    'sliding_windows' => [5m, 15m, 1h, 6h, 24h, 7d],
    'burst_ratio_threshold' => 3.0,
    'cross_account' => [enabled, shared_device_threshold=3, shared_ip_threshold=5],
]
'geolocation' => [
    'impossible_travel_max_speed_kmh' => 900.0,
    'ip_reputation_threshold' => 0.6,
    'high_risk_countries' => env('FRAUD_HIGH_RISK_COUNTRIES'),
    'geo_cluster' => [eps_km=50, min_points=3, max_points=1000, max_distance=500km],
]
'batch' => [schedule=hourly, chunk_size=100, lookback_hours=24]
```

### Test Structure (115 tests)
- `AnomalyDetectionOrchestratorTest` (11): Integration of all 4 detection types
- `StatisticalAnalysisServiceTest` (12): Z-score, IQR, IF, LOF, seasonal
- `BehavioralAnalysisEnhancementsTest` (10): Adaptive thresholds, drift, segmentation
- `VelocityEnhancementsTest` (10): Sliding windows, burst, cross-account
- `GeolocationEnhancementsTest` (8): IP reputation, impossible travel, clustering
- `GeoMathServiceTest` (12): Haversine, DBSCAN, clustering
- `AnomalyEdgeCaseTest` (21): Division by zero, boundaries, null handling, geo extremes
- `AnomalyDetectionTest` (8): Model factory, severity, status
- `AnomalyEnumsTest` (6): Enum values and methods
- `ProcessAnomalyBatchJobTest` (6): Batch processing
- `FraudDetectionServiceTest` (8): Integration
- `FraudDetectedTest` (3): Events
